<?php

/*
 * Plugin Name:       Team Hydra Login
 * Description:       This plugin allows you to register and login to Wordpress using your Team Hydra account.
 * Version:           3.12.0
 * Requires at least: 5.0
 * Requires PHP:      7.4
 * Author:            Team Hydra
 * Author URI:        https://teamhydra.dev
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       team-hydra-login
 * Domain Path:       /languages
 */
/**
 * Team Hydra Account Login
 *
 *
 * */

/*
Notes
  Spec Doc - http://openid.net/specs/team-hydra-account-basic-1_0-32.html

  Filters
  - team-hydra-account-alter-request       - 3 args: request array, plugin settings, specific request op
  - team-hydra-account-settings-fields     - modify the fields provided on the settings page
  - team-hydra-account-login-button-text   - modify the login button text
  - team-hydra-account-cookie-redirect-url - modify the redirect url stored as a cookie
  - team-hydra-account-user-login-test     - (bool) should the user be logged in based on their claim
  - team-hydra-account-user-creation-test  - (bool) should the user be created based on their claim
  - team-hydra-account-auth-url            - modify the authentication url
  - team-hydra-account-alter-user-claim    - modify the user_claim before a new user is created
  - team-hydra-account-alter-user-data     - modify user data before a new user is created
  - team-hydra-account-modify-token-response-before-validation - modify the token response before validation
  - team-hydra-account-modify-id-token-claim-before-validation - modify the token claim before validation

  Actions
  - team-hydra-account-user-create                     - 2 args: fires when a new user is created by this plugin
  - team-hydra-account-user-update                     - 1 arg: user ID, fires when user is updated by this plugin
  - team-hydra-account-update-user-using-current-claim - 2 args: fires every time an existing user logs in and the claims are updated.
  - team-hydra-account-redirect-user-back              - 2 args: $redirect_url, $user. Allows interruption of redirect during login.
  - team-hydra-account-user-logged-in                  - 1 arg: $user, fires when user is logged in.
  - team-hydra-account-cron-daily                      - daily cron action
  - team-hydra-account-state-not-found                 - the given state does not exist in the database, regardless of its expiration.
  - team-hydra-account-state-expired                   - the given state exists, but expired before this login attempt.

  Callable actions

  User Meta
  - team-hydra-account-subject-identity    - the identity of the user provided by the idp
  - team-hydra-account-last-id-token-claim - the user's most recent id_token claim, decoded
  - team-hydra-account-last-user-claim     - the user's most recent user_claim
  - team-hydra-account-last-token-response - the user's most recent token response

  Options
  - team_hydra_account_settings     - plugin settings
  - team-hydra-account-valid-states - locally stored generated states
*/


/**
 * Team_Hydra_Account class.
 *
 * Defines plugin initialization functionality.
 *
 * @package Team_Hydra_Account
 * @category  General
 */
class Team_Hydra_Account {

	/**
	 * Singleton instance of self
	 *
	 * @var Team_Hydra_Account
	 */
	protected static $_instance = null;

	/**
	 * Plugin version.
	 *
	 * @var string
	 */
	const VERSION = '3.12.0';

	/**
	 * Plugin settings.
	 *
	 * @var Team_Hydra_Account_Option_Settings
	 */
	private $settings;

	/**
	 * Plugin logs.
	 *
	 * @var Team_Hydra_Account_Option_Logger
	 */
	private $logger;

	/**
	 *
	 * @var Team_Hydra_Account_Client
	 */
	private $client;

	/**
	 * Client wrapper.
	 *
	 * @var Team_Hydra_Account_Client_Wrapper
	 */
	public $client_wrapper;

	/**
	 * Setup the plugin
	 *
	 * @param Team_Hydra_Account_Option_Settings $settings The settings object.
	 * @param Team_Hydra_Account_Option_Logger   $logger   The loggin object.
	 *
	 * @return void
	 */
	public function __construct( Team_Hydra_Account_Option_Settings $settings, Team_Hydra_Account_Option_Logger $logger ) {
		$this->settings = $settings;
		$this->logger = $logger;
		self::$_instance = $this;
	}

	// @codeCoverageIgnoreStart

	/**
	 * WordPress Hook 'init'.
	 *
	 * @return void
	 */
	public function init() {

		$this->client = new Team_Hydra_Account_Client(
			$this->settings->client_id,
			$this->settings->client_secret,
			$this->settings->scope,
			$this->settings->endpoint_login,
			$this->settings->endpoint_userinfo,
			$this->settings->endpoint_token,
			$this->get_redirect_uri( $this->settings ),
			$this->settings->acr_values,
			$this->get_state_time_limit( $this->settings ),
			$this->logger
		);

		$this->client_wrapper = Team_Hydra_Account_Client_Wrapper::register( $this->client, $this->settings, $this->logger );
		if ( defined( 'WP_CLI' ) && WP_CLI ) {
			return;
		}

		Team_Hydra_Account_Login_Form::register( $this->settings, $this->client_wrapper );

		// Add a shortcode to get the auth URL.
		add_shortcode( 'team_hydra_account_auth_url', array( $this->client_wrapper, 'get_authentication_url' ) );

		// Add actions to our scheduled cron jobs.
		add_action( 'team-hydra-account-cron-daily', array( $this, 'cron_states_garbage_collection' ) );

		$this->upgrade();

		if ( is_admin() ) {
			Team_Hydra_Account_Settings_Page::register( $this->settings, $this->logger );
		}
	}

	/**
	 * Get the default redirect URI.
	 *
	 * @param Team_Hydra_Account_Option_Settings $settings The settings object.
	 *
	 * @return string
	 */
	public function get_redirect_uri( Team_Hydra_Account_Option_Settings $settings ) {
		$redirect_uri = admin_url( 'admin-ajax.php?action=team-hydra-account-authorize' );

		if ( $settings->alternate_redirect_uri ) {
			$redirect_uri = site_url( '/team-hydra-account-authorize' );
		}

		return $redirect_uri;
	}

	/**
	 * Get the default state time limit.
	 *
	 * @param Team_Hydra_Account_Option_Settings $settings The settings object.
	 *
	 * @return int
	 */
	public function get_state_time_limit( Team_Hydra_Account_Option_Settings $settings ) {
		$state_time_limit = 180;
		// State time limit cannot be zero.
		if ( $settings->state_time_limit ) {
			$state_time_limit = intval( $settings->state_time_limit );
		}

		return $state_time_limit;
	}

	/**
	 * Check if privacy enforcement is enabled, and redirect users that aren't
	 * logged in.
	 *
	 * @return void
	 */
	public function enforce_privacy_redirect() {
		if ( $this->settings->enforce_privacy && ! is_user_logged_in() ) {
			// The client endpoint relies on the wp-admin ajax endpoint.
			if (
				! defined( 'DOING_AJAX' ) ||
				! boolval( constant( 'DOING_AJAX' ) ) ||
				! isset( $_GET['action'] ) ||
				'team-hydra-account-authorize' != $_GET['action'] ) {
				auth_redirect();
			}
		}
	}

	/**
	 * Enforce privacy settings for rss feeds.
	 *
	 * @param string $content The content.
	 *
	 * @return mixed
	 */
	public function enforce_privacy_feeds( $content ) {
		if ( $this->settings->enforce_privacy && ! is_user_logged_in() ) {
			$content = __( 'Private site', 'team-hydra-account' );
		}
		return $content;
	}

	/**
	 * Handle plugin upgrades
	 *
	 * @return void
	 */
	public function upgrade() {
		$last_version = get_option( 'team-hydra-account-plugin-version', 0 );
		$settings = $this->settings;

		if ( version_compare( self::VERSION, $last_version, '>' ) ) {
			// An upgrade is required.
			self::setup_cron_jobs();

			// @todo move this to another file for upgrade scripts
			if ( isset( $settings->ep_login ) ) {
				$settings->endpoint_login = $settings->ep_login;
				$settings->endpoint_token = $settings->ep_token;
				$settings->endpoint_userinfo = $settings->ep_userinfo;

				unset( $settings->ep_login, $settings->ep_token, $settings->ep_userinfo );
				$settings->save();
			}

			// Update the stored version number.
			update_option( 'team-hydra-account-plugin-version', self::VERSION );
		}
	}

	/**
	 * Expire state transients by attempting to access them and allowing the
	 * transient's own mechanisms to delete any that have expired.
	 *
	 * @return void
	 */
	public function cron_states_garbage_collection() {
		global $wpdb;
		$states = $wpdb->get_col( "SELECT `option_name` FROM {$wpdb->options} WHERE `option_name` LIKE '_transient_team-hydra-account-state--%'" );

		if ( ! empty( $states ) ) {
			foreach ( $states as $state ) {
				$transient = str_replace( '_transient_', '', $state );
				get_transient( $transient );
			}
		}
	}

	/**
	 * Ensure cron jobs are added to the schedule.
	 *
	 * @return void
	 */
	public static function setup_cron_jobs() {
		if ( ! wp_next_scheduled( 'team-hydra-account-cron-daily' ) ) {
			wp_schedule_event( time(), 'daily', 'team-hydra-account-cron-daily' );
		}
	}

	/**
	 * Activation hook.
	 *
	 * @return void
	 */
	public static function activation() {
		self::setup_cron_jobs();
	}

	/**
	 * Deactivation hook.
	 *
	 * @return void
	 */
	public static function deactivation() {
		wp_clear_scheduled_hook( 'team-hydra-account-cron-daily' );
	}

	/**
	 * Simple autoloader.
	 *
	 * @param string $class The class name.
	 *
	 * @return void
	 */
	public static function autoload( $class ) {
		$prefix = 'Team_Hydra_Account_';

		if ( stripos( $class, $prefix ) !== 0 ) {
			return;
		}

		$filename = $class . '.php';

		// Internal files are all lowercase and use dashes in filenames.
		if ( false === strpos( $filename, '\\' ) ) {
			$filename = strtolower( str_replace( '_', '-', $filename ) );
		} else {
			$filename  = str_replace( '\\', DIRECTORY_SEPARATOR, $filename );
		}

		$filepath = __DIR__ . '/includes/' . $filename;

		if ( file_exists( $filepath ) ) {
			require_once $filepath;
		}
	}

	/**
	 * Instantiate the plugin and hook into WordPress.
	 *
	 * @return void
	 */
	public static function bootstrap() {
		/**
		 * This is a documented valid call for spl_autoload_register.
		 *
		 * @link https://www.php.net/manual/en/function.spl-autoload-register.php#71155
		 */
		spl_autoload_register( array( 'Team_Hydra_Account', 'autoload' ) );

		$settings = new Team_Hydra_Account_Option_Settings(
			// Default settings values.
			array(
				// OAuth client settings.
				'login_type'           => defined( 'OIDC_LOGIN_TYPE' ) ? OIDC_LOGIN_TYPE : 'auto',
				'client_id'            => defined( 'OIDC_CLIENT_ID' ) ? OIDC_CLIENT_ID : '',
				'client_secret'        => defined( 'OIDC_CLIENT_SECRET' ) ? OIDC_CLIENT_SECRET : '',
				'scope'                => defined( 'OIDC_CLIENT_SCOPE' ) ? OIDC_CLIENT_SCOPE : 'email profile openid',
				'endpoint_login'       => defined( 'OIDC_ENDPOINT_LOGIN_URL' ) ? OIDC_ENDPOINT_LOGIN_URL : 'https://accounts.hep.gg/application/o/authorize/',
				'endpoint_userinfo'    => defined( 'OIDC_ENDPOINT_USERINFO_URL' ) ? OIDC_ENDPOINT_USERINFO_URL : 'https://accounts.hep.gg/application/o/userinfo/',
				'endpoint_token'       => defined( 'OIDC_ENDPOINT_TOKEN_URL' ) ? OIDC_ENDPOINT_TOKEN_URL : 'https://accounts.hep.gg/application/o/token/',
				'endpoint_end_session' => defined( 'OIDC_ENDPOINT_LOGOUT_URL' ) ? OIDC_ENDPOINT_LOGOUT_URL : '',
				'acr_values'           => defined( 'OIDC_ACR_VALUES' ) ? OIDC_ACR_VALUES : '',

				// Non-standard settings.
				'no_sslverify'           => 0,
				'http_request_timeout'   => 5,
				'identity_key'           => 'sub',
				'nickname_key'           => 'preferred_username',
				'email_format'           => '{email}',
				'displayname_format'     => '',
				'identify_with_username' => false,
				'state_time_limit'       => 180,

				// Plugin settings.
				'enforce_privacy'          => defined( 'OIDC_ENFORCE_PRIVACY' ) ? intval( OIDC_ENFORCE_PRIVACY ) : 0,
				'alternate_redirect_uri'   => 1,
				'token_refresh_enable'     => 1,
				'link_existing_users'      => defined( 'OIDC_LINK_EXISTING_USERS' ) ? intval( OIDC_LINK_EXISTING_USERS ) : 1,
				'create_if_does_not_exist' => defined( 'OIDC_CREATE_IF_DOES_NOT_EXIST' ) ? intval( OIDC_CREATE_IF_DOES_NOT_EXIST ) : 1,
				'redirect_user_back'       => defined( 'OIDC_REDIRECT_USER_BACK' ) ? intval( OIDC_REDIRECT_USER_BACK ) : 1,
				'redirect_on_logout'       => defined( 'OIDC_REDIRECT_ON_LOGOUT' ) ? intval( OIDC_REDIRECT_ON_LOGOUT ) : 1,
				'enable_logging'           => defined( 'OIDC_ENABLE_LOGGING' ) ? intval( OIDC_ENABLE_LOGGING ) : 0,
				'log_limit'                => defined( 'OIDC_LOG_LIMIT' ) ? intval( OIDC_LOG_LIMIT ) : 1000,
			)
		);

		$logger = new Team_Hydra_Account_Option_Logger( 'error', $settings->enable_logging, $settings->log_limit );

		$plugin = new self( $settings, $logger );

		add_action( 'init', array( $plugin, 'init' ) );

		// Privacy hooks.
		add_action( 'template_redirect', array( $plugin, 'enforce_privacy_redirect' ), 0 );
		add_filter( 'the_content_feed', array( $plugin, 'enforce_privacy_feeds' ), 999 );
		add_filter( 'the_excerpt_rss', array( $plugin, 'enforce_privacy_feeds' ), 999 );
		add_filter( 'comment_text_rss', array( $plugin, 'enforce_privacy_feeds' ), 999 );
	}

	/**
	 * Create (if needed) and return a singleton of self.
	 *
	 * @return Team_Hydra_Account
	 */
	public static function instance() {
		if ( null === self::$_instance ) {
			self::bootstrap();
		}
		return self::$_instance;
	}
}

Team_Hydra_Account::instance();

register_activation_hook( __FILE__, array( 'Team_Hydra_Account', 'activation' ) );
register_deactivation_hook( __FILE__, array( 'Team_Hydra_Account', 'deactivation' ) );

// Provide publicly accessible plugin helper functions.
require_once 'includes/functions.php';
